//
// (c) 2020 wesolutions GmbH
// All rights reserved.
//

import QtQuick 2.12

import wesual.Ui 1.0

import wesual.Ui.Private 1.0

UiToolButtonBase {
    id : uiCheckButton

    property alias iconName : icon.iconName
    property alias iconSize : icon.iconSize
    property alias text     : label.text
    property alias tooltip  : tooltip.text

    /*!
    \brief Defines the checked state of the control where a value of \c true
    indicates that the box is currently checked.

    \nobinding
    This behavior can be changed by setting \l bindable to \c true.
    */
    property bool checked : false

    /*!
    \brief
    Disables the internal handling of the checked state to make the
    \l checked property bindable.

    When this property is set to true, the control will only emit the
    \l {UiButtonBase::}{clicked} signal but will not modify the checked
    property internally.
    This can be used the bind an external boolean property to the checked
    state that is modified in response to the
    \l {UiButtonBase::}{clicked} signal.
    */
    property bool bindable : false

    implicitHeight   : icon.implicitHeight + 6
    implicitWidth    : {
        if (label.visible) {
            return icon.implicitWidth + label.implicitWidth + 17;
        }

        return icon.implicitWidth + 6
    }
    activeFocusOnTab : false
    baselineOffset   : label.y + label.baselineOffset

    onClicked : {
        if (!bindable) {
            checked = !checked;
        }
    }

    UiTooltip { id : tooltip }

    Rectangle {
        id : background

        color        : UiColors.getColor(UiColors.White)
        anchors.fill : parent
        visible      : false

        states : [
            State {
                name : "disabled"
                when : !uiCheckButton.enabled

                PropertyChanges {
                    target  : background
                    color   : UiColors.getColor(UiColors.DisabledLightGrey13)
                    visible : checked
                }
            },
            State {
                name : "pressed"
                when : uiCheckButton.pressed

                PropertyChanges {
                    target  : background
                    color   : UiColors.getColor(UiColors.SelectionGreen)
                    visible : true
                }
            },
            State {
                name   : "checked"
                when   : uiCheckButton.checked
                extend : "pressed"
            },
            State {
                name : "hovered"
                when : uiCheckButton.hovered || uiCheckButton.activeFocus

                PropertyChanges {
                    target  : background
                    color   : UiColors.getColor(UiColors.HoverGreen)
                    visible : true
                }
            }
        ]
    }

    UiIcon {
        id : icon

        x : 3
        y : 3

        anchors.centerIn :
            (uiCheckButton.style === Ui.IconOnly || text.length === 0)
            ? parent : undefined

        states : [
            State {
                name : "disabled"
                when : !uiCheckButton.enabled

                PropertyChanges {
                    target   : icon
                    disabled : 1.0
                }
            },
            State {
                name : "highlight"
                when : uiCheckButton.pressed || uiCheckButton.checked

                PropertyChanges {
                    target    : icon
                    highlight : 1.0
                }
            }
        ]
    }

    UiText {
        id : label

        visible   : text.length > 0 && uiCheckButton.style !== Ui.IconOnly
        highlight : uiCheckButton.pressed || uiCheckButton.checked
        width     : uiCheckButton.width - icon.implicitWidth - 17

        anchors {
            verticalCenter : icon.verticalCenter
            left           : icon.right
            leftMargin     : 5
        }
    }
}

